#include "pairing.h"

static void curve_group_init(element_ptr e)
{
    e->data = malloc(sizeof(point_t));
    point_init(e->data, ((curve_group_data_ptr) e->field->data)->curve);
}

static void curve_group_clear(element_ptr e)
{
    point_clear(e->data);
    free(e->data);
}

static void curve_group_mul(element_ptr x, element_ptr a, element_ptr b)
{
    point_add(x->data, a->data, b->data);
}

static void curve_group_set1(element_ptr x)
{
    point_set_inf(x->data);
}

static void curve_group_set(element_ptr x, element_ptr y)
{
    point_set(x->data, y->data);
}

static void curve_group_random(element_ptr x)
{
    curve_group_data_ptr p = x->field->data;
    point_random(x->data);
    point_mul(x->data, p->cofac, x->data);
}

static void curve_group_from_hash(element_ptr x, int len, void *data)
{
    curve_group_data_ptr p = x->field->data;
    point_from_hash(x->data, len, data);
    point_mul(x->data, p->cofac, x->data);
}

static void curve_group_out_str(FILE *stream, int base, element_ptr x)
{
    point_out_str(stream, base, x->data);
}

static int curve_group_length_in_bytes(element_ptr x)
{
    point_ptr P = x->data;
    return element_length_in_bytes(P->x) + element_length_in_bytes(P->y);
}

void field_clear_curve_group(field_t f)
{
    curve_group_data_ptr p;
    p = f->data;
    mpz_clear(p->cofac);
    free(p);
}

void field_init_curve_group(field_t f, curve_t c, mpz_t cofac)
{
    curve_group_data_ptr p;
    p = f->data = malloc(sizeof(curve_group_data_t));
    p->curve = c;
    mpz_init(p->cofac);
    mpz_set(p->cofac, cofac);
    f->init = curve_group_init;
    f->clear = curve_group_clear;
    f->mul = curve_group_mul;
    f->set1 = curve_group_set1;
    f->set = curve_group_set;
    f->random = curve_group_random;
    f->from_hash = curve_group_from_hash;
    f->out_str = curve_group_out_str;
    f->field_clear = field_clear_curve_group;
    if (c->field->fixed_length_in_bytes < 0) {
	f->length_in_bytes = curve_group_length_in_bytes;
    } else {
	f->fixed_length_in_bytes = 2 * c->field->fixed_length_in_bytes;
    }
}

static void cc_pairing(element_ptr out, element_ptr in1, element_ptr in2,
	pairing_t pairing)
{
    mnt_pairing_data_ptr p = pairing->data;
    cc_miller(out, pairing->r, in1->data, in2->data);
    element_pow(out, out, p->tateexp);
}

static int cc_is_almost_coddh(element_ptr a, element_ptr b,
	element_ptr c, element_ptr d,
	pairing_t pairing)
{
    int res = 0;
    element_t t0, t1, t2;

    element_init(t0, pairing->GT);
    element_init(t1, pairing->GT);
    element_init(t2, pairing->GT);
    mnt_pairing_data_ptr p = pairing->data;
    cc_miller(t0, pairing->r, a->data, d->data);
    cc_miller(t1, pairing->r, b->data, c->data);
    element_pow(t0, t0, p->tateexp);
    element_pow(t1, t1, p->tateexp);
    element_mul(t2, t0, t1);
    if (element_is1(t2)) {
	//g, g^x, h, h^-x case
	res = 1;
    } else {
	element_invert(t1, t1);
	element_mul(t2, t0, t1);
	if (element_is1(t2)) {
	    //g, g^x, h, h^x case
	    res = 1;
	}
    }
    element_clear(t0);
    element_clear(t1);
    element_clear(t2);
    return res;
}

void pairing_init_cc_param(pairing_t pairing, cc_param_t param)
{
    mnt_pairing_data_ptr p;
    element_t a, b;
    element_t irred;
    mpz_t z;

    mpz_init(pairing->r);
    mpz_set(pairing->r, param->r);
    field_init_fp(pairing->Zr, pairing->r);
    pairing->map = cc_pairing;
    pairing->is_almost_coddh = cc_is_almost_coddh;

    p =	pairing->data = malloc(sizeof(mnt_pairing_data_t));
    field_init_fp(p->Fq, param->q);
    element_init(a, p->Fq);
    element_init(b, p->Fq);
    element_set_mpz(a, param->a);
    element_set_mpz(b, param->b);
    curve_init_cc_ab(p->Eq, a, b);

    field_init_poly(p->Fqx, p->Fq);
    element_init(irred, p->Fqx);
    do {
	poly_random_monic(irred, param->k);
    } while (!poly_is_irred(irred));
    field_init_polymod(p->Fqk, irred);
    element_clear(irred);

    mpz_init(p->tateexp);
    mpz_sub_ui(p->tateexp, p->Fqk->order, 1);
    mpz_divexact(p->tateexp, p->tateexp, pairing->r);

    cc_init_extend(p->Eqk, p->Eq, p->Fqk);
    field_init_curve_group(pairing->G1, p->Eq, param->h);
    mpz_init(z);
    mpz_set_si(z, 1);
    field_init_curve_group(pairing->G2, p->Eqk, z);
    mpz_clear(z);
    pairing->GT = p->Fqk;
}
