//pairing-based signatures library
#include "sig.h"

void bls_sign(unsigned char *sig, int hashlen, void *hash,
	bls_private_key_t sk)
{
    element_t h;
    point_ptr P;

    element_init(h, sk->param->pairing->G1);
    element_from_hash(h, hashlen, hash);
    element_pow(h, h, sk->x);
    //unfortunately we must give up the element_t abstraction
    //since we only want to output the x coordinate
    P = h->data;
    element_to_bytes(sig, P->x);

    element_clear(h);
}

int bls_verify(unsigned char *sig, int hashlen, void *hash,
	bls_public_key_t pk)
{
    //have to mess with internals since we are only given the x-coord
    element_t x;
    element_t hx;
    element_t h;
    int res;

    field_ptr G1 = pk->param->pairing->G1;
    curve_group_data_ptr p = G1->data;

    element_init(h, G1);
    element_from_hash(h, hashlen, hash);
    element_init(hx, G1);
    element_init(x, p->curve->field);
    element_from_bytes(x, sig);
    point_from_x((point_ptr) hx->data, x);

    res = is_almost_coddh(h, hx, pk->param->g, pk->gx, pk->param->pairing);
    element_clear(x);
    element_clear(hx);
    return res;
}

void bls_gen_sys_param(bls_sys_param_t param, pairing_t pairing)
{
    param->pairing = pairing;
    element_init(param->g, pairing->G2);
    element_random(param->g);
    //signature is only one coordinate
    param->signature_length = pairing->G1->fixed_length_in_bytes / 2;
}

void bls_gen(bls_public_key_t pk, bls_private_key_t sk, bls_sys_param_t param)
{
    pk->param = sk->param = param;
    mpz_init(sk->x);
    element_init(pk->gx, param->pairing->G2);
    pbc_mpz_random(sk->x, param->pairing->r);
    element_pow(pk->gx, param->g, sk->x);
}

void bb_gen_sys_param(bb_sys_param_t param, pairing_t pairing)
{
    param->pairing = pairing;
    //signature is a point (only need one coordinate) and an element of Z_r
    param->signature_length = pairing->G1->fixed_length_in_bytes + mpz_sizeinbase(pairing->r, 2);
}

void bb_gen(bb_public_key_t pk, bb_private_key_t sk, bb_sys_param_t param)
{
    pairing_ptr pairing = param->pairing;
    pk->param = sk->param = param;
    
    element_init(sk->x, pairing->Zr);
    element_init(sk->y, pairing->Zr);
    element_random(sk->x);
    element_random(sk->y);
    element_init(pk->g1, param->pairing->G1);
    element_init(pk->g2, param->pairing->G2);
    element_init(pk->z, param->pairing->GT);
    element_random(pk->g2);
    //TODO: use trace map to find g1
    element_random(pk->g1);
    element_init(pk->u, param->pairing->G2);
    element_init(pk->v, param->pairing->G2);
    element_pow(pk->u, pk->g2, sk->x->data);
    element_pow(pk->v, pk->g2, sk->y->data);
    bilinear_map(pk->z, pk->g1, pk->g2, param->pairing);
}

void bb_sign(unsigned char *sig, int hashlen, void *hash,
	bb_public_key_t pk, bb_private_key_t sk)
{
    element_t sigma;
    point_ptr P;
    element_t r, z, m;
    bb_sys_param_ptr param = pk->param;
    pairing_ptr pairing = param->pairing;

    element_init(r, pairing->Zr);
    element_init(z, pairing->Zr);
    element_init(m, pairing->Zr);

    element_random(r);
    mpz_import(m->data, hashlen, 1, 1, 1, 0, hash);
    mpz_mod(m->data, m->data, pairing->r);
    element_mul(z, sk->y, r);
    element_add(z, z, sk->x);
    element_add(z, z, m);
    element_invert(z, z);
    element_init(sigma, pairing->G1);
    element_pow(sigma, pk->g1, z->data);

    //unfortunately we must give up the element_t abstraction
    //since we only want to output the x coordinate
    P = sigma->data;
    element_to_bytes(sig, P->x);
    element_to_bytes(sig + element_length_in_bytes(P->x), r);

    element_clear(sigma);
    element_clear(r);
    element_clear(z);
    element_clear(m);
}

int bb_verify(unsigned char *sig, int hashlen, void *hash,
	bb_public_key_t pk)
{
    //have to mess with internals since we are only given the x-coord
    element_t sigma, r;
    element_t m;
    element_t t0, t1, t2;
    element_t x;
    int res;
    pairing_ptr pairing = pk->param->pairing;

    field_ptr G1 = pairing->G1;
    curve_group_data_ptr p = G1->data;

    element_init(m, pairing->Zr);
    mpz_import(m->data, hashlen, 1, 1, 1, 0, hash);
    mpz_mod(m->data, m->data, pairing->r);

    element_init(x, p->curve->field);
    element_from_bytes(x, sig);

    element_init(r, pairing->Zr);
    element_from_bytes(r, sig + element_length_in_bytes(x));

    element_init(sigma, pairing->G1);
    point_from_x((point_ptr) sigma->data, x);

    element_init(t0, pairing->G2);
    element_init(t1, pairing->G2);
    element_init(t2, pairing->GT);

    element_pow(t0, pk->g2, m->data);
    element_pow(t1, pk->v, r->data);
    element_mul(t0, t0, t1);
    element_mul(t0, t0, pk->u);
    bilinear_map(t2, sigma, t0, pairing);
    if (!element_cmp(t2, pk->z)) {
	res = 1;
    } else {
	element_mul(t2, t2, pk->z);
	res = element_is1(t2);
    }

    element_clear(t0);
    element_clear(t1);
    element_clear(t2);
    element_clear(m);
    element_clear(sigma);
    element_clear(r);
    element_clear(x);
    return res;
}
