//Find MNT curves with embedding degree 6

#include <math.h> //needed for floor, sqrt
#include <stdlib.h>
#include "mpc.h"
#include "pbc.h"

void cm_info_init(cm_info_t cm)
{
    mpz_init(cm->q);
    mpz_init(cm->r);
    mpz_init(cm->h);
    mpz_init(cm->n);
}

void cm_info_clear(cm_info_t cm)
{
    mpz_clear(cm->q);
    mpz_clear(cm->r);
    mpz_clear(cm->h);
    mpz_clear(cm->n);
}

static mpf_t pi, eulere;

void mpc_exp(mpc_t res, mpc_t z, int iter)
{
    //res = exp(z) = 1 + z/1! + z^2/2! + ...
    //converges quickly around the origin
    mpc_t z0, z1;
    mpf_t f0;
    int i;

    mpc_init(z0);
    mpc_init(z1);
    mpf_init(f0);

    i=1;
    mpc_set_ui(z0, 1);
    mpc_set_ui(z1, 1);
    for(;i<iter;) {
	mpc_mul(z0, z0, z);
	mpf_set_ui(f0, 1);
	mpf_div_ui(f0, f0, i);
	mpc_mul_mpf(z0, z0, f0);
	mpc_add(z1, z1, z0);
	i++;
    }
//printf("z0 = ");
//mpc_out_str(stdout, 10, 80, z0);
//printf("\n");

    mpc_set(res, z1);
    mpc_clear(z0);
    mpc_clear(z1);
    mpf_clear(f0);
}

static void compute_q(mpc_t q, mpc_t tau)
    //q = exp(2 pi i tau)
{
    mpc_t z0;
    mpf_t f0, f1;
    mpf_ptr fp0;
    unsigned long pwr;

    mpc_init(z0);
    mpf_init(f0);
    mpf_init(f1);

    //compute z0 = 2 pi i tau
    mpc_set(z0, tau);
    //first remove integral part of Re(tau)
    //since exp(2 pi i)  = 1
    //it seems |Re(tau)| < 1 anyway?
    fp0 = mpc_re(z0);
    mpf_trunc(f1, fp0);
    mpf_sub(fp0, fp0, f1);

    mpc_mul_mpf(z0, z0, pi);
    mpc_mul_ui(z0, z0, 2);
    mpc_muli(z0, z0);

    //compute q = exp(z0);
    //first write z0 = A + a + b i
    //where A is an integer
    //a, b are in [-1, 1]
    //then compute e^A separately
    //(perhaps computing e^a separately may be more
    //efficient too?)
    fp0 = mpc_re(z0);
    pwr = mpf_get_ui(fp0);
    mpf_ui_div(f0, 1, eulere);
    mpf_pow_ui(f0, f0, pwr);
    mpf_add_ui(fp0, fp0, pwr);

    fp0 = mpc_im(z0);
    pwr = mpf_get_ui(fp0);
    mpc_exp(q, z0, 1000);
    mpc_mul_mpf(q, q, f0);
    mpc_clear(z0);
    mpf_clear(f0);
    mpf_clear(f1);
}

static void compute_Delta(mpc_t z, mpc_t q)
    //z = Delta(q) (see Cohen)
{
    int d;
    int n;
    int pow;
    mpc_t z0, z1, z2;

    mpc_init(z0);
    mpc_init(z1);
    mpc_init(z2);

    mpc_set_ui(z0, 1);
    d = -1;
    for(n=1; n<100; n++) {
	pow = n *(3 * n - 1) / 2;
	mpc_pow_ui(z1, q, pow);
	mpc_pow_ui(z2, q, n);
	mpc_mul(z2, z2, z1);
	mpc_add(z1, z1, z2);
	if (d) {
	    mpc_sub(z0, z0, z1);
	    d = 0;
	} else {
	    mpc_add(z0, z0, z1);
	    d = 1;
	}
    }

    mpc_pow_ui(z0, z0, 24);
    mpc_mul(z, z0, q);

    mpc_clear(z0);
    mpc_clear(z1);
    mpc_clear(z2);
}

static void compute_h(mpc_t z, mpc_t tau)
    //z = h(tau)
    //(called h() by Blake et al, f() by Cohen)
{
    mpc_t z0, z1, q;
    mpc_init(q);
    mpc_init(z0);
    mpc_init(z1);
    compute_q(q, tau);
    mpc_mul(z0, q, q);
    compute_Delta(z0, z0);
    compute_Delta(z1, q);
    mpc_div(z, z0, z1);
    mpc_clear(q);
    mpc_clear(z0);
    mpc_clear(z1);
}

static void compute_j(mpc_t j, mpc_t tau)
    //j = j(tau)
{
    mpc_t h;
    mpc_t z0;
    mpc_init(h);
    mpc_init(z0);
    compute_h(h, tau);
    //TODO: use mul 2exp functions
    mpc_mul_ui(z0, h, 256);
    mpc_add_ui(z0, z0, 1);
    mpc_pow_ui(z0, z0, 3);
    mpc_div(j, z0, h);
    mpc_clear(z0);
    mpc_clear(h);
}

static void set_precision(int prec)
{
    //printf("prec: %d\n", prec);
    mpf_set_default_prec(prec);
    mpf_init(pi);
    mpf_init(eulere);
    {
	int i;
	mpf_t f0;

	mpf_init(f0);
	mpf_set_ui(eulere, 1);
	mpf_set_ui(f0, 1);
	for (i=1; i<prec; i++) {
	    mpf_div_ui(f0, f0, i);
	    mpf_add(eulere, eulere, f0);
	}
	mpf_clear(f0);
    }

    //TODO: compute this up to required accuracy instead
    mpf_set_str(pi,
    "3. \
141592653589793238462643383279502884197169399375105820974944 \
592307816406286208998628034825342117067982148086513282306647 \
093844609550582231725359408128481117450284102701938521105559 \
644622948954930381964428810975665933446128475648233786783165 \
271201909145648566923460348610454326648213393607260249141273 \
724587006606315588174881520920962829254091715364367892590360 \
011330530548820466521384146951941511609433057270365759591953 \
092186117381932611793105118548074462379962749567351885752724 \
891227938183011949129833673362440656643086021394946395224737 \
190702179860943702770539217176293176752384674818467669405132 \
000568127145263560827785771342757789609173637178721468440901 \
224953430146549585371050792279689258923542019956112129021960 \
864034418159813629774771309960518707211349999998372978049951 \
059731732816096318595024459455346908302642522308253344685035 \
261931188171010003137838752886587533208381420617177669147303 \
598253490428755468731159562863882353787593751957781857780532 \
171226806613001927876611195909216420198938095257201065485863 \
278865936153381827968230301952035301852968995773622599413891 \
249721775283479131515574857242454150695950829533116861727855 \
889075098381754637464939319255060400927701671139009848824012 \
858361603563707660104710181942955596198946767837449448255379 \
774726847104047534646208046684259069491293313677028989152104 \
752162056966024058038150193511253382430035587640247496473263 \
914199272604269922796782354781636009341721641219924586315030 \
286182974555706749838505494588586926995690927210797509302955 \
321165344987202755960236480665499119881834797753566369807426 \
542527862551818417574672890977772793800081647060016145249192 \
173217214772350141441973568548161361157352552133475741849468 \
438523323907394143334547762416862518983569485562099219222184 \
272550254256887671790494601653466804988627232791786085784383 \
827967976681454100953883786360950680064225125205117392984896 \
084128488626945604241965285022210661186306744278622039194945 \
047123713786960956364371917287467764657573962413890865832645 \
995813390478027590099465764078951269468398352595709825822620 \
522489407726719478268482601476990902640136394437455305068203 \
496252451749399651431429809190659250937221696461515709858387 \
410597885959772975498930161753928468138268683868942774155991 \
855925245953959431049972524680845987273644695848653836736222 \
626099124608051243884390451244136549762780797715691435997700 \
129616089441694868555848406353422072225828488648158456028506 \
016842739452267467678895252138522549954666727823986456596116 \
354886230577456498035593634568174324112515076069479451096596 \
094025228879710893145669136867228748940560101503308617928680 \
920874760917824938589009714909675985261365549781893129784821 \
682998948722658804857564014270477555132379641451523746234364 \
542858444795265867821051141354735739523113427166102135969536 \
231442952484937187110145765403590279934403742007310578539062 \
198387447808478489683321445713868751943506430218453191048481 \
005370614680674919278191197939952061419663428754440643745123 \
718192179998391015919561814675142691239748940907186494231961 \
567945208095146550225231603881930142093762137855956638937787 \
" ,10);
}

void hilbert_poly(darray_t P, int D)
    //returns darray of mpz's
    //that are coefficients of H_D(x)
    //(see Cohen)
{
    int a;
    int b = D % 2;
    int t;
    int B = floor(sqrt(D/3));
    mpc_t alpha;
    mpc_t j;
    mpf_t sqrtD;
    mpf_t f0;
    darray_t Pz;
    mpc_t z0, z1, z2;

    set_precision(B * 30 + 10);

    darray_init(Pz);
    mpc_init(alpha);
    mpc_init(j);
    mpc_init(z0);
    mpc_init(z1);
    mpc_init(z2);
    mpf_init(sqrtD);
    mpf_init(f0);

    mpf_sqrt_ui(sqrtD, D);
    for (;;) {
	t = (b*b + D) / 4;
	if (b > 1) {
	    a = b;
	} else {
	    a = 1;
	}
step3:
	if (t % a) {
step4:
	    a++;
	    if (a * a <= t) goto step3;
	} else {
	    //a, b, t/a are coeffs of an appropriate
	    //primitive reduced positive definite form
	    //compute j((-b + sqrt{-D})/(2a))
//printf("a b c = %d %d %d\n", a, b, t/a);
	    mpf_set_ui(f0, 1);
	    mpf_div_ui(f0, f0, 2 * a);
	    mpf_mul(mpc_im(alpha), sqrtD, f0);
	    mpf_mul_ui(f0, f0, b);
	    mpf_neg(mpc_re(alpha), f0);

	    compute_j(j, alpha);
if (0) {
    int i;
    for (i=Pz->count - 1; i>=0; i--) {
	printf("P %d = ", i);
	mpc_out_str(stdout, 10, 4, Pz->item[i]);
	printf("\n");
    }
}

	    if (a == b || a * a == t || !b) {
		//P *= X - j
		int i, n;
		mpc_ptr p0;
		p0 = (mpc_ptr) malloc(sizeof(mpc_t));
		mpc_init(p0);
		mpc_neg(p0, j);
		n = Pz->count;
		if (n) {
		    mpc_set(z1, Pz->item[0]);
		    mpc_add(Pz->item[0], z1, p0);
		    for (i=1; i<n; i++) {
			mpc_mul(z0, z1, p0);
			mpc_set(z1, Pz->item[i]);
			mpc_add(Pz->item[i], z1, z0);
		    }
		    mpc_mul(p0, p0, z1);
		}
		darray_append(Pz, p0);
	    } else {
		//P *= X^2 - 2 Re(j) X + |j|^2
		int i, n;
		mpc_ptr p0, p1;
		p0 = (mpc_ptr) malloc(sizeof(mpc_t));
		p1 = (mpc_ptr) malloc(sizeof(mpc_t));
		mpc_init(p0);
		mpc_init(p1);
		//p1 = - 2 Re(j)
		mpf_mul_ui(f0, mpc_re(j), 2);
		mpf_neg(f0, f0);
		mpf_set(mpc_re(p1), f0);
		//p0 = |j|^2
		mpf_mul(f0, mpc_re(j), mpc_re(j));
		mpf_mul(mpc_re(p0), mpc_im(j), mpc_im(j));
		mpf_add(mpc_re(p0), mpc_re(p0), f0);
		n = Pz->count;
		if (!n) {
		} else if (n == 1) {
		    mpc_set(z1, Pz->item[0]);
		    mpc_add(Pz->item[0], z1, p1);
		    mpc_mul(p1, z1, p1);
		    mpc_add(p1, p1, p0);
		    mpc_mul(p0, p0, z1);
		} else {
		    mpc_set(z2, Pz->item[0]);
		    mpc_set(z1, Pz->item[1]);
		    mpc_add(Pz->item[0], z2, p1);
		    mpc_mul(z0, z2, p1);
		    mpc_add(Pz->item[1], z1, z0);
		    mpc_add(Pz->item[1], Pz->item[1], p0);
		    for (i=2; i<n; i++) {
			mpc_mul(z0, z1, p1);
			mpc_mul(alpha, z2, p0);
			mpc_set(z2, z1);
			mpc_set(z1, Pz->item[i]);
			mpc_add(alpha, alpha, z0);
			mpc_add(Pz->item[i], z1, alpha);
		    }
		    mpc_mul(z0, z2, p0);
		    mpc_mul(p1, p1, z1);
		    mpc_add(p1, p1, z0);
		    mpc_mul(p0, p0, z1);
		}
		darray_append(Pz, p1);
		darray_append(Pz, p0);
	    }
	    goto step4;
	}
	b+=2;
	if (b > B) break;
    }

    //round polynomial
    {
	int i;
	mpz_ptr coeff;
	for (i=Pz->count - 1; i>=0; i--) {
	    coeff = (mpz_ptr) malloc(sizeof(mpz_t));
	    mpz_init(coeff);
	    if (mpf_sgn(mpc_re(Pz->item[i])) < 0) {
		mpf_set_d(f0, -0.5);
	    } else {
		mpf_set_d(f0, 0.5);
	    }
	    mpf_add(f0, f0, mpc_re(Pz->item[i]));
	    mpz_set_f(coeff, f0);
	    darray_append(P, coeff);
	    mpc_clear(Pz->item[i]);
	    free(Pz->item[i]);
	}
	coeff = (mpz_ptr) malloc(sizeof(mpz_t));
	mpz_init(coeff);
	mpz_set_ui(coeff, 1);
	darray_append(P, coeff);
    }
    darray_clear(Pz);
    mpc_clear(z0);
    mpc_clear(z1);
    mpc_clear(z2);
    mpf_clear(f0);
    mpf_clear(sqrtD);
    mpc_clear(alpha);
    mpc_clear(j);
    mpf_clear(pi);
}

int findroot(element_ptr root, element_ptr poly)
    //returns 0 if a root exists and sets root to one of the roots
    //otherwise return value is nonzero
{
    //compute gcd(x^q - x, poly)
    //don't need to ensure sqfreeness?
    field_t fpxmod;
    element_t p, x, r, fac;
    mpz_t q;
    
    mpz_init(q);
    mpz_set(q, poly_base_field(poly)->order);

    field_init_polymod(fpxmod, poly);
    element_init(p, fpxmod);
    element_init(x, fpxmod);
    poly_setx(x); 
    /*
    printf("q = ");
    mpz_out_str(stdout, 10, q);
    printf("\n");
    */
    element_pow(p, x, q);
    element_sub(p, p, x);

    poly_unmod(p);
    poly_gcd(p, p, poly);
    poly_make_monic(p, p);
    element_clear(x);
    field_clear(fpxmod);

    if (!poly_degree(p)) {
	printf("no roots!\n");
	mpz_clear(q);
	element_clear(p);
	return -1;
    }
    //Use Cantor-Zassenhaus to find a root
    element_init(fac, p->field);
    element_init(x, p->field);
    element_set_si(x, 1);
    mpz_sub_ui(q, q, 1);
    mpz_divexact_ui(q, q, 2);
    for (;;) {
	if (poly_degree(p) == 1) {
	    //found a root!
	    break;
	}
	field_init_polymod(fpxmod, p);
	element_init(r, fpxmod);
step_random:
	element_random(r);
	r->field = p->field;
	poly_gcd(fac, r, p);
	r->field = fpxmod;
	if (poly_degree(fac) > 0) {
	    poly_make_monic(p, fac);
	} else {
	    int n;
	    element_pow(r, r, q);
	    r->field = p->field;
	    element_add(r, r, x);
	    poly_gcd(fac, r, p);
	    r->field = fpxmod;
	    n = poly_degree(fac);
	    if (n > 0 && n < poly_degree(p)) {
		poly_make_monic(p, fac);
	    } else {
		goto step_random;
	    }
	}
	element_clear(r);
	field_clear(fpxmod);
    }
    element_neg(root, poly_coeff(p, 0));
    mpz_clear(q);
    element_clear(x);
    element_clear(p);
    element_clear(fac);
    return 0;
}

int mnt_step2(darray_ptr L, int D, mpz_t U)
{
    int d;
    mpz_t n, l, q;
    mpz_t p;
    mpz_t r, cofac;
    cm_info_ptr cm;

    mpz_init(l);
    mpz_mod_ui(l, U, 6);
    if (!mpz_cmp_ui(l, 1)) {
	mpz_sub_ui(l, U, 1);
	d = 1;
    } else if (!mpz_cmp_ui(l, 5)) {
	mpz_add_ui(l, U, 1);
	d = -1;
    } else {
	mpz_clear(l);
	return 1;
    }

    mpz_divexact_ui(l, l, 3);
    mpz_init(q);

    mpz_mul(q, l, l);
    mpz_add_ui(q, q, 1);
    if (!mpz_probab_prime_p(q, 10)) {
	mpz_clear(q);
	mpz_clear(l);
	return 1;
    }

    mpz_init(n);
    if (d < 0) {
	mpz_sub(n, q, l);
    } else {
	mpz_add(n, q, l);
    }


    mpz_init(p);
    mpz_init(r);
    mpz_init(cofac);
{
    mpz_set_ui(cofac, 1);
    mpz_set(r, n);
    mpz_set_ui(p, 2);
    if (!mpz_probab_prime_p(r, 10)) for(;;) {
	if (mpz_divisible_p(r, p)) do {
	    mpz_mul(cofac, cofac, p);
	    mpz_divexact(r, r, p);
	} while (mpz_divisible_p(r, p));
	if (mpz_probab_prime_p(r, 10)) break;
	//TODO: use a table of primes instead?
	mpz_nextprime(p, p);
	if (mpz_sizeinbase(p, 2) > 16) {
	    //printf("has 16+ bit factor\n");
	    mpz_clear(r);
	    mpz_clear(p);
	    mpz_clear(cofac);
	    mpz_clear(q);
	    mpz_clear(l);
	    mpz_clear(n);
	    return 1;
	}
    }
}

    cm = malloc(sizeof(cm_info_t));
    cm_info_init(cm);
    cm->k = 6;
    cm->D = D;
    mpz_set(cm->q, q);
    mpz_set(cm->r, r);
    mpz_set(cm->h, cofac);
    mpz_set(cm->n, n);
    darray_append(L, cm);

    mpz_clear(cofac);
    mpz_clear(r);
    mpz_clear(p);
    mpz_clear(q);
    mpz_clear(l);
    mpz_clear(n);
    return 0;
}

int find_mnt6_curve(darray_t L, int D, int bitlimit)
{
    //first solve DV^2 = 3l^2 - 2l + 3
    //use variable subst U = 3l - 1 to transform equation
    //U^2 - 3DV^2 = -8
    //so 3D cannot be a square
    //(the only squares that differ by 8 are 1 and 9,
    //which we get if U=V=1, D=3, but then l is not an integer)

    //a0, twice_a0 don't change once initialized
    //a1 is a_i every iteration
    //P0, P1 become P_{i-1}, P_i every iteration
    //similarly for Q0, Q1
    mpz_t a0, twice_a0, a1;
    mpz_t P0, P1;
    mpz_t Q0, Q1;
    //variables to compute the convergents
    mpz_t p0, p1, pnext;
    mpz_t q0, q1, qnext;

    int d;
    int found_count = 0;

    mpz_t D3;
    
    darray_t listp, listq;
    mpz_ptr zptr;

    mpz_init(a0); mpz_init(twice_a0); mpz_init(a1);
    mpz_init(P0); mpz_init(P1);
    mpz_init(Q0); mpz_init(Q1);
    mpz_init(p0); mpz_init(p1); mpz_init(pnext);
    mpz_init(q0); mpz_init(q1); mpz_init(qnext);
    mpz_init(D3);
    mpz_set_ui(D3, 3 * D);

    darray_init(listp);
    darray_init(listq);

    if (mpz_perfect_square_p(D3)) {
	return 0;
    }
    mpz_sqrt(a0, D3);
    mpz_set_ui(P0, 0);
    mpz_set_ui(Q0, 1);

    mpz_set(P1, a0);
    mpz_mul(Q1, a0, a0);
    mpz_sub(Q1, D3, Q1);
    mpz_add(a1, a0, P1);
    mpz_tdiv_q(a1, a1, Q1);

    mpz_add(twice_a0, a0, a0);

    mpz_set(p0, a0);
    mpz_set_ui(q0, 1);
    mpz_mul(p1, a0, a1);
    mpz_add_ui(p1, p1, 1);
    mpz_set(q1, a1);

    //mpz_out_str(stdout, 10, p0);
    //printf(" / ");
    //mpz_out_str(stdout, 10, q0);
    //printf("\n");

    d = -1;
    for(;;) {
	//printf("Q: ");
	if (d == -1) {
	    //printf("-");
	    if (!mpz_cmp_ui(Q1, 8)) {
		zptr = (mpz_ptr) malloc(sizeof(mpz_t));
		mpz_init(zptr);
		mpz_set(zptr, p0);
		darray_append(listp, zptr);
		zptr = (mpz_ptr) malloc(sizeof(mpz_t));
		mpz_init(zptr);
		mpz_set(zptr, q0);
		darray_append(listq, zptr);
	    } else if (!mpz_cmp_ui(Q1, 2)) {
		zptr = (mpz_ptr) malloc(sizeof(mpz_t));
		mpz_init(zptr);
		mpz_mul_ui(zptr, p0, 2);
		darray_append(listp, zptr);
		zptr = (mpz_ptr) malloc(sizeof(mpz_t));
		mpz_init(zptr);
		mpz_mul_ui(zptr, q0, 2);
		darray_append(listq, zptr);
	    }
	}
	//mpz_out_str(stdout, 10, Q1);
	//printf("\n");
	//mpz_out_str(stdout, 10, p1);
	//printf(" / ");
	//mpz_out_str(stdout, 10, q1);
	//printf("\n");
	if (!mpz_cmp(twice_a0, a1)) break;
	//compute more of the continued fraction expansion
	mpz_set(P0, P1);
	mpz_mul(P1, a1, Q1);
	mpz_sub(P1, P1, P0);
	mpz_set(Q0, Q1);
	mpz_mul(Q1, P1, P1);
	mpz_sub(Q1, D3, Q1);
	mpz_divexact(Q1, Q1, Q0);
	mpz_add(a1, a0, P1);
	mpz_tdiv_q(a1, a1, Q1);

	//compute next convergent
	mpz_mul(pnext, a1, p1);
	mpz_add(pnext, pnext, p0);
	mpz_set(p0, p1);
	mpz_set(p1, pnext);

	mpz_mul(qnext, a1, q1);
	mpz_add(qnext, qnext, q0);
	mpz_set(q0, q1);
	mpz_set(q1, qnext);
	d = -d;
    }

    {
	int i, n;
	n = listp->count;
	for (;;) {
	    for (i=0; i<n; i++) {
		/*
		mpz_out_str(stdout, 0, listp->item[i]);
		printf(", ");
		mpz_out_str(stdout, 0, listq->item[i]);
		printf("\n");
		*/
		if (!mnt_step2(L, D, listp->item[i])) found_count++;
		//compute next solution as follows
		//if p, q is current solution
		//compute new solution p', q' via
		//(p + q sqrt{3D})(t + u sqrt{3D}) = p' + q' sqrt(3D)
		//where t, u is min. solution to Pell equation
		//can use a0, p1, q1 as temp variables now
		mpz_mul(p1, p0, listp->item[i]);
		mpz_mul(q1, q0, listq->item[i]);
		mpz_mul(q1, q1, D3);
		mpz_add(p1, p1, q1);
		if (2 * mpz_sizeinbase(p1, 2) > bitlimit + 10) goto toobig;
		mpz_mul(a0, p0, listq->item[i]);
		mpz_mul(q1, q0, listp->item[i]);
		mpz_add(a0, a0, q1);
		mpz_set(listp->item[i], p1);
		mpz_set(listq->item[i], a0);
	    }
	}
    }
toobig:

    mpz_clear(a0); mpz_clear(twice_a0); mpz_clear(a1);
    mpz_clear(P0); mpz_clear(P1);
    mpz_clear(Q0); mpz_clear(Q1);
    mpz_clear(p0); mpz_clear(p1); mpz_clear(pnext);
    mpz_clear(q0); mpz_clear(q1); mpz_clear(qnext);
    mpz_clear(D3);
    return found_count;
}

void compute_cm_curve(cc_param_ptr param, cm_info_ptr cm)
    //computes a curve and sets fp to the field it is defined over
    //using the complex multiplication method, where cm holds
    //the appropriate information (e.g. discriminant, field order)
{
    darray_t coefflist;
    element_t hp, root;
    field_t fp, fpx;
    int i, n;
    curve_t cc;

    field_init_fp(fp, cm->q);
    field_init_poly(fpx, fp);
    element_init(hp, fpx);

    darray_init(coefflist);

    hilbert_poly(coefflist, cm->D);

    if (0) {
	n = coefflist->count;
	printf("x^%d", n - 1);
	printf("\n");
	for (i=n-2; i>=0; i--) {
	    mpz_out_str(stdout, 10, coefflist->item[i]);
	    if (i) {
		if (i == 1) {
		    printf("x");
		} else {
		    printf("x^%d", i);
		}
	    }
	    printf("\n");
	}
    }

    n = coefflist->count;
    poly_alloc(hp, n);
    for (i=0; i<n; i++) {
	element_set_mpz(poly_coeff(hp, i), coefflist->item[i]);
    }
    //TODO: memory leak: clear elements of coefflist
    darray_clear(coefflist);
    //TODO: remove x = 0, 1728 roots
    //TODO: what if there's no roots?
    /*
    printf("hp ");
    element_out_str(stdout, hp);
    printf("\n");
    */
    element_init(root, fp);
    findroot(root, hp);
    //printf("root = ");
    //element_out_str(stdout, root);
    //printf("\n");
    element_clear(hp);
    field_clear(fpx);

    //the root is the j-invariant of our desired curve
    curve_init_cc_j(cc, root);
    element_clear(root);

    //we may need to twist it however
    {
	point_t P;

	//pick a random point P and see if it has the right order
	point_init(P, cc);
	point_random(P);
	point_mul(P, cm->n, P);
	//printf("P = ");
	//point_out_str(stdout, P);
	//printf("\n");
	//if not, we twist the curve
	if (!point_is_inf(P)) {
	    twist_curve(cc);
	}
	point_clear(P);
    }

    mpz_set(param->q, cm->q);
    mpz_set(param->n, cm->n);
    mpz_set(param->h, cm->h);
    mpz_set(param->r, cm->r);
    mpz_set(param->a, ((common_curve_ptr) cc->data)->a->data);
    mpz_set(param->b, ((common_curve_ptr) cc->data)->b->data);
    param->k = cm->k;
    {
	mpz_t z;
	mpz_init(z);
	//compute order of curve in F_q^k
	//n = q - t + 1 hence t = q - n + 1
	mpz_sub(z, param->q, param->n);
	mpz_add_ui(z, z, 1);
	compute_trace_n(z, param->q, z, param->k);
	mpz_pow_ui(param->nk, param->q, param->k);
	mpz_sub_ui(z, z, 1);
	mpz_sub(param->nk, param->nk, z);
	mpz_mul(z, param->r, param->r);
	mpz_divexact(param->hk, param->nk, z);
	mpz_clear(z);
    }
    curve_clear(cc);
    field_clear(fp);
}
