#include "pairing.h"
#include "solinas.h"

//assumes P is in the base field, Q in some field extension,
//and will be used in Tate pairing computation
static void cc_miller(element_t res, mpz_t q, point_t P,
	element_ptr Qx, element_ptr Qy, fieldmap mapbase)
{
    //collate divisions
    int m;
    element_t v, vd;
    point_t Z;
    element_t a, b, c;
    common_curve_ptr cc = P->curve->data;
    element_t t0;
    element_t e0, e1;

    void do_vertical(element_t e)
    {
	if (point_is_inf(Z)) {
	    return;
	}
	mapbase(e0, Z->x);
	element_sub(e0, Qx, e0);
	element_mul(e, e, e0);
    }

    void do_tangent(element_t e)
    {
	//a = -slope_tangent(Z.x, Z.y);
	//b = 1;
	//c = -(Z.y + a * Z.x);
	//but we multiply by 2*Z.y to avoid division

	//a = -Zx * (Zx + Zx + Zx + twicea_2) - a_4;
	//Common curves: a2 = 0 (and cc->a is a_4), so
	//a = -(Zx (Zx + Zx + Zx) + cc->a)
	//b = 2 * Zy
	//c = -(2 Zy^2 + a Zx);
	element_ptr Zx = Z->x;
	element_ptr Zy = Z->y;

	if (point_is_inf(Z)) {
	    return;
	}

	if (element_is0(Zy)) {
	    //order 2
	    do_vertical(e);
	    return;
	}

	element_add(a, Zx, Zx);
	element_add(a, a, Zx);
	element_mul(a, a, Zx);
	element_add(a, a, cc->a);
	element_neg(a, a);

	element_add(b, Zy, Zy);

	element_mul(t0, b, Zy);
	element_mul(c, a, Zx);
	element_add(c, c, t0);
	element_neg(c, c);

	//TODO: use poly_mul_constant?
	mapbase(e0, a);
	element_mul(e0, e0, Qx);
	mapbase(e1, b);
	element_mul(e1, e1, Qy);
	element_add(e0, e0, e1);
	mapbase(e1, c);
	element_add(e0, e0, e1);
	element_mul(e, e, e0);
    }

    void do_line(element_ptr e)
    {
	//a = -(B.y - A.y) / (B.x - A.x);
	//b = 1;
	//c = -(A.y + a * A.x);
	//but we'll multiply by B.x - A.x to avoid division

	element_ptr Ax = Z->x;
	element_ptr Ay = Z->y;
	element_ptr Bx = P->x;
	element_ptr By = P->y;

	//we assume B is never O
	if (point_is_inf(Z)) {
	    do_vertical(e);
	    return;
	}

	if (!element_cmp(Ax, Bx)) {
	    if (!element_cmp(Ay, By)) {
		do_tangent(e);
		return;
	    }
	    do_vertical(e);
	    return;
	}

	element_sub(b, Bx, Ax);
	element_sub(a, Ay, By);
	element_mul(t0, b, Ay);
	element_mul(c, a, Ax);
	element_add(c, c, t0);
	element_neg(c, c);

	mapbase(e0, a);
	element_mul(e0, e0, Qx);
	mapbase(e1, b);
	element_mul(e1, e1, Qy);
	element_add(e0, e0, e1);
	mapbase(e1, c);
	element_add(e0, e0, e1);
	element_mul(e, e, e0);
    }

    element_init(a, P->curve->field);
    element_init(b, P->curve->field);
    element_init(c, P->curve->field);
    element_init(t0, P->curve->field);
    element_init(e0, res->field);
    element_init(e1, res->field);

    element_init(v, res->field);
    element_init(vd, res->field);
    point_init(Z, P->curve);

    point_set(Z, P);

    element_set1(v);
    element_set1(vd);
    m = mpz_sizeinbase(q, 2) - 2;

    while(m >= 0) {
	element_mul(v, v, v);
	element_mul(vd, vd, vd);
	do_tangent(v);
	point_double(Z, Z);
	do_vertical(vd);
	if (mpz_tstbit(q, m)) {
	    do_line(v);
	    point_add(Z, Z, P);
	    do_vertical(vd);
	}
	m--;
    }

    element_invert(vd, vd);
    element_mul(res, v, vd);

    element_clear(v);
    element_clear(vd);
    point_clear(Z);
    element_clear(a);
    element_clear(b);
    element_clear(c);
    element_clear(t0);
    element_clear(e0);
    element_clear(e1);
}

static void cc_tatepower(element_ptr out, element_ptr in, pairing_t pairing)
{
    mnt_pairing_data_ptr p = pairing->data;
    //TODO: this should only be called by odd k routines, so remove
    //following section after even k case is fast enough
    if (p->k == 6) {
	element_t e0, e1, e2, e3;
	element_init(e0, out->field);
	element_init(e1, out->field);
	element_init(e2, out->field);
	element_init(e3, out->field);
	void subexpr(element_ptr e) {
	    element_set0(e0);
	    poly_set_coeff(e0, poly_coeff(in, 0), 0);
	    poly_const_mul(e2, poly_coeff(in, 1), e);
	    element_add(e0, e0, e2);
	    element_mul(e1, e, e);
	    poly_const_mul(e2, poly_coeff(in, 2), e1);
	    element_add(e0, e0, e2);
	    element_mul(e1, e1, e);
	    poly_const_mul(e2, poly_coeff(in, 3), e1);
	    element_add(e0, e0, e2);
	    element_mul(e1, e1, e);
	    poly_const_mul(e2, poly_coeff(in, 4), e1);
	    element_add(e0, e0, e2);
	    element_mul(e1, e1, e);
	    poly_const_mul(e2, poly_coeff(in, 5), e1);
	    element_add(e0, e0, e2);
	}
	subexpr(p->xpowq[4]);
	element_set(e3, e0);
	subexpr(p->xpowq[3]);
	element_mul(e3, e3, e0);
	subexpr(p->xpowq[1]);
	element_mul(e0, e0, in);
	element_invert(e0, e0);
	element_mul(out, e3, e0);
	element_pow(out, out, p->tateexp);
	element_clear(e0);
	element_clear(e1);
	element_clear(e2);
	element_clear(e3);
    } else {
	element_pow(out, in, p->tateexp);
    }
}

static void cc_pairing(element_ptr out, element_ptr in1, element_ptr in2,
	pairing_t pairing)
{
    mnt_pairing_data_ptr p = pairing->data;
    point_ptr Q = in2->data;
    cc_miller(out, pairing->r, in1->data, Q->x, Q->y, p->mapbase);
    cc_tatepower(out, out, pairing);
}

static int cc_is_almost_coddh_odd_k(element_ptr a, element_ptr b,
	element_ptr c, element_ptr d,
	pairing_t pairing)
{
    int res = 0;
    element_t t0, t1, t2;

    element_init(t0, pairing->GT);
    element_init(t1, pairing->GT);
    element_init(t2, pairing->GT);
    mnt_pairing_data_ptr p = pairing->data;
    point_ptr Pc = c->data;
    point_ptr Pd = d->data;
    cc_miller(t0, pairing->r, a->data, Pd->x, Pd->y, p->mapbase);
    cc_miller(t1, pairing->r, b->data, Pc->x, Pc->y, p->mapbase);
    cc_tatepower(t1, t1, pairing);
    cc_tatepower(t0, t0, pairing);
    element_mul(t2, t0, t1);
    if (element_is1(t2)) {
	//g, g^x, h, h^-x case
	res = 1;
    } else {
	element_invert(t1, t1);
	element_mul(t2, t0, t1);
	if (element_is1(t2)) {
	    //g, g^x, h, h^x case
	    res = 1;
	}
    }
    element_clear(t0);
    element_clear(t1);
    element_clear(t2);
    return res;
}

static void trace(element_ptr out, element_ptr in, pairing_ptr pairing)
{
    int i;
    point_ptr p = in->data;
    point_ptr r = out->data;
    point_t q;
    mnt_pairing_data_ptr mpdp = pairing->data;

    point_init(q, p->curve);

    point_set(q, p);
    point_set(r, p);

    for (i=1; i<mpdp->k; i++) {
	cc_frobenius(q, q, mpdp->Fq->order);
	point_add(r, r, q);
    }
    point_clear(q);
}

static void pairing_init_cc_param_odd_k(pairing_t pairing, cc_param_t param)
{
    mnt_pairing_data_ptr p;
    element_t a, b;
    element_t irred;
    mpz_t z;

    mpz_init(pairing->r);
    mpz_set(pairing->r, param->r);
    field_init_fp(pairing->Zr, pairing->r);
    pairing->map = cc_pairing;
    pairing->is_almost_coddh = cc_is_almost_coddh_odd_k;

    p =	pairing->data = malloc(sizeof(mnt_pairing_data_t));
    field_init_fp(p->Fq, param->q);
    element_init(a, p->Fq);
    element_init(b, p->Fq);
    element_set_mpz(a, param->a);
    element_set_mpz(b, param->b);
    curve_init_cc_ab(p->Eq, a, b);

    field_init_poly(p->Fqx, p->Fq);
    element_init(irred, p->Fqx);
    do {
	poly_random_monic(irred, param->k);
    } while (!poly_is_irred(irred));
    field_init_polymod(p->Fqk, irred);
    element_clear(irred);

    if (param->k == 6) {
	//TODO: remove this eventually, once even k case is fast enough
	int i;
	element_t e0;
	mpz_ptr q = param->q;
	mpz_ptr z = p->tateexp;
	mpz_init(z);
	mpz_mul(z, q, q);
	mpz_sub(z, z, q);
	mpz_add_ui(z, z, 1);
	mpz_divexact(z, z, pairing->r);

	p->xpowq = malloc(sizeof(element_ptr) * 5);
	element_init(e0, p->Fqk);
	poly_setx(e0);
	for (i=1; i<=4; i++) {
	    element_ptr e = p->xpowq[i] = malloc(sizeof(element_t));
	    element_init(e, p->Fqk);
	    element_pow(e0, e0, q);
	    element_set(e, e0);
	}
	element_clear(e0);
    } else {
	mpz_init(p->tateexp);
	mpz_sub_ui(p->tateexp, p->Fqk->order, 1);
	mpz_divexact(p->tateexp, p->tateexp, pairing->r);
    }

    p->mapbase = ((polymod_field_data_ptr) p->Fqk->data)->mapbase;

    cc_init_map_curve(p->Eqk, p->Eq, p->Fqk, p->mapbase);

    pairing->G1 = malloc(sizeof(field_t));
    pairing->G2 = malloc(sizeof(field_t));

    field_init_curve_group(pairing->G1, p->Eq, param->h);
    mpz_init(z);
    mpz_set_si(z, 1);
    field_init_curve_group(pairing->G2, p->Eqk, z);
    mpz_clear(z);
    p->k = param->k;
    pairing->GT = p->Fqk;
    pairing->phi = trace;

    element_clear(a);
    element_clear(b);
}

static void cc_miller_no_denom(element_t res, mpz_t q, point_t P,
	element_ptr Qx, element_ptr Qy, fieldmap mapbase)
{
    //collate divisions
    int m;
    element_t v;
    point_t Z;
    element_t a, b, c;
    common_curve_ptr cc = P->curve->data;
    element_t t0;
    element_t e0, e1;

    void do_vertical(void)
    {
	if (point_is_inf(Z)) {
	    return;
	}
	mapbase(e0, Z->x);
	element_sub(e0, Qx, e0);
	element_mul(v, v, e0);
    }

    void do_tangent(void)
    {
	//a = -slope_tangent(Z.x, Z.y);
	//b = 1;
	//c = -(Z.y + a * Z.x);
	//but we multiply by 2*Z.y to avoid division

	//a = -Zx * (Zx + Zx + Zx + twicea_2) - a_4;
	//Common curves: a2 = 0 (and cc->a is a_4), so
	//a = -(Zx (Zx + Zx + Zx) + cc->a)
	//b = 2 * Zy
	//c = -(2 Zy^2 + a Zx);
	element_ptr Zx = Z->x;
	element_ptr Zy = Z->y;

	if (point_is_inf(Z)) {
	    return;
	}

	if (element_is0(Zy)) {
	    //order 2
	    do_vertical();
	    return;
	}

	element_add(a, Zx, Zx);
	element_add(a, a, Zx);
	element_mul(a, a, Zx);
	element_add(a, a, cc->a);
	element_neg(a, a);

	element_add(b, Zy, Zy);

	element_mul(t0, b, Zy);
	element_mul(c, a, Zx);
	element_add(c, c, t0);
	element_neg(c, c);

	//TODO: use poly_mul_constant?
	mapbase(e0, a);
	element_mul(e0, e0, Qx);
	mapbase(e1, b);
	element_mul(e1, e1, Qy);
	element_add(e0, e0, e1);
	mapbase(e1, c);
	element_add(e0, e0, e1);
	element_mul(v, v, e0);
    }

    void do_line(void)
    {
	//a = -(B.y - A.y) / (B.x - A.x);
	//b = 1;
	//c = -(A.y + a * A.x);
	//but we'll multiply by B.x - A.x to avoid division

	element_ptr Ax = Z->x;
	element_ptr Ay = Z->y;
	element_ptr Bx = P->x;
	element_ptr By = P->y;

	//we assume B is never O
	if (point_is_inf(Z)) {
	    do_vertical();
	    return;
	}

	if (!element_cmp(Ax, Bx)) {
	    if (!element_cmp(Ay, By)) {
		do_tangent();
		return;
	    }
	    do_vertical();
	    return;
	}

	element_sub(b, Bx, Ax);
	element_sub(a, Ay, By);
	element_mul(t0, b, Ay);
	element_mul(c, a, Ax);
	element_add(c, c, t0);
	element_neg(c, c);

	mapbase(e0, a);
	element_mul(e0, e0, Qx);
	mapbase(e1, b);
	element_mul(e1, e1, Qy);
	element_add(e0, e0, e1);
	mapbase(e1, c);
	element_add(e0, e0, e1);
	element_mul(v, v, e0);
    }

    element_init(a, P->curve->field);
    element_init(b, P->curve->field);
    element_init(c, P->curve->field);
    element_init(t0, P->curve->field);
    element_init(e0, res->field);
    element_init(e1, res->field);

    element_init(v, res->field);
    point_init(Z, P->curve);

    point_set(Z, P);

    element_set1(v);
    m = mpz_sizeinbase(q, 2) - 2;

    while(m >= 0) {
	element_mul(v, v, v);
	do_tangent();
	point_double(Z, Z);
	if (mpz_tstbit(q, m)) {
	    do_line();
	    point_add(Z, Z, P);
	}
	m--;
    }

    element_set(res, v);

    element_clear(v);
    point_clear(Z);
    element_clear(a);
    element_clear(b);
    element_clear(c);
    element_clear(t0);
    element_clear(e0);
    element_clear(e1);
}

static void cc_tatepower_even_k(element_ptr out, element_ptr in, pairing_t pairing)
{
    even_mnt_pairing_data_ptr p = pairing->data;
    if (p->k == 6) {
	element_t e0, e1, e2, e3;
	element_init(e0, p->Fqk);
	element_init(e1, p->Fqd);
	element_init(e2, p->Fqd);
	element_init(e3, p->Fqk);
	element_ptr e0re = fi_re(e0);
	element_ptr e0im = fi_im(e0);
	element_ptr inre = fi_re(in);
	element_ptr inim = fi_im(in);
	void subexpr(element_ptr e) {
	    element_set0(e0);
	    poly_set_coeff(e0re, poly_coeff(inre, 0), 0);
	    poly_set_coeff(e0im, poly_coeff(inim, 0), 0);
	    poly_const_mul(e2, poly_coeff(inre, 1), e);
	    element_add(e0re, e0re, e2);
	    poly_const_mul(e2, poly_coeff(inim, 1), e);
	    element_add(e0im, e0im, e2);
	    element_mul(e1, e, e);
	    poly_const_mul(e2, poly_coeff(inre, 2), e1);
	    element_add(e0re, e0re, e2);
	    poly_const_mul(e2, poly_coeff(inim, 2), e1);
	    element_add(e0im, e0im, e2);
	}
	subexpr(p->xpowq[4]);
	element_set(e3, e0);
	subexpr(p->xpowq[3]);
	element_neg(e0im, e0im);
	element_mul(e3, e3, e0);
	subexpr(p->xpowq[1]);
	element_neg(e0im, e0im);
	element_mul(e0, e0, in);
	element_invert(e0, e0);
	element_mul(out, e3, e0);
	element_pow(out, out, p->tateexp);
	element_clear(e0);
	element_clear(e1);
	element_clear(e2);
	element_clear(e3);
    } else {
	element_pow(out, in, p->tateexp);
    }
}

static void cc_pairing_even_k(element_ptr out, element_ptr in1, element_ptr in2,
	pairing_t pairing)
{
    point_ptr Qbase = in2->data;
    element_t x, y;
    even_mnt_pairing_data_ptr p = pairing->data;

    element_init(x, out->field);
    element_init(y, out->field);
    //map from twist: (x, y) --> (v^-1 x, v^-(3/2) y)
    //where v is the quadratic nonresidue used to construct the twist
    element_mul(fi_re(x), Qbase->x, p->nqrinv);
    //v^-3/2 = v^-2 * v^1/2
    element_mul(fi_im(y), Qbase->y, p->nqrinv2);
    cc_miller_no_denom(out, pairing->r, in1->data, x, y, p->mapbase);
    cc_tatepower_even_k(out, out, pairing);
}

static int cc_is_almost_coddh_even_k(element_ptr a, element_ptr b,
	element_ptr c, element_ptr d,
	pairing_t pairing)
{
    int res = 0;
    element_t t0, t1, t2;
    element_t cx, cy;
    element_t dx, dy;
    even_mnt_pairing_data_ptr p = pairing->data;

    element_init(cx, p->Fqk);
    element_init(cy, p->Fqk);
    element_init(dx, p->Fqk);
    element_init(dy, p->Fqk);

    element_init(t0, pairing->GT);
    element_init(t1, pairing->GT);
    element_init(t2, pairing->GT);
    point_ptr Pc = c->data;
    point_ptr Pd = d->data;
    //map from twist: (x, y) --> (v^-1 x, v^-(3/2) y)
    //where v is the quadratic nonresidue used to construct the twist
    element_mul(fi_re(cx), Pc->x, p->nqrinv);
    element_mul(fi_re(dx), Pd->x, p->nqrinv);
    //v^-3/2 = v^-2 * v^1/2
    element_mul(fi_im(cy), Pc->y, p->nqrinv2);
    element_mul(fi_im(dy), Pd->y, p->nqrinv2);

    cc_miller_no_denom(t0, pairing->r, a->data, dx, dy, p->mapbase);
    cc_miller_no_denom(t1, pairing->r, b->data, cx, cy, p->mapbase);
    cc_tatepower_even_k(t0, t0, pairing);
    cc_tatepower_even_k(t1, t1, pairing);
    element_mul(t2, t0, t1);
    if (element_is1(t2)) {
	//g, g^x, h, h^-x case
	res = 1;
    } else {
	element_invert(t1, t1);
	element_mul(t2, t0, t1);
	if (element_is1(t2)) {
	    //g, g^x, h, h^x case
	    res = 1;
	}
    }
    element_clear(t0);
    element_clear(t1);
    element_clear(t2);
    return res;
}

static void Fq_to_Fqd_to_Fqk(element_t out, element_t in)
{
    element_field_to_poly(fi_re(out), in);
    element_set0(fi_im(out));
}

static void pairing_init_cc_param_even_k(pairing_t pairing, cc_param_t param)
{
    even_mnt_pairing_data_ptr p;
    element_t a, b;
    element_t irred;
    mpz_t z;
    int d = param->k / 2;

    mpz_init(pairing->r);
    mpz_set(pairing->r, param->r);
    field_init_fp(pairing->Zr, pairing->r);
    pairing->map = cc_pairing_even_k;
    pairing->is_almost_coddh = cc_is_almost_coddh_even_k;

    p =	pairing->data = malloc(sizeof(even_mnt_pairing_data_t));
    field_init_fp(p->Fq, param->q);
    element_init(a, p->Fq);
    element_init(b, p->Fq);
    element_set_mpz(a, param->a);
    element_set_mpz(b, param->b);
    curve_init_cc_ab(p->Eq, a, b);

    field_init_poly(p->Fqx, p->Fq);
    element_init(irred, p->Fqx);
    do {
	poly_random_monic(irred, d);
    } while (!poly_is_irred(irred));
    field_init_polymod(p->Fqd, irred);
    element_clear(irred);

    //find a quadratic nonresidue of Fqd lying in Fq
    //TODO: assuming exists an element in Fq that is not a square in Fqd
    //assert(!p->Fqd->nqr);
    p->Fqd->nqr = malloc(sizeof(element_t));
    element_init(p->Fqd->nqr, p->Fqd);
    do {
	poly_alloc(p->Fqd->nqr, 1);
	element_random(poly_coeff(p->Fqd->nqr, 0));
    } while (element_is_sqr(p->Fqd->nqr));

    field_init_quadratic(p->Fqk, p->Fqd);

    if (param->k == 6) {
	int i;
	element_t e0;
	mpz_ptr q = param->q;
	mpz_ptr z = p->tateexp;
	mpz_init(z);
	mpz_mul(z, q, q);
	mpz_sub(z, z, q);
	mpz_add_ui(z, z, 1);
	mpz_divexact(z, z, pairing->r);

	p->xpowq = malloc(sizeof(element_ptr) * 5);
	element_init(e0, p->Fqd);
	poly_setx(e0);
	for (i=1; i<=4; i++) {
	    element_ptr e = p->xpowq[i] = malloc(sizeof(element_t));
	    element_init(e, p->Fqd);
	    element_pow(e0, e0, q);
	    element_set(e, e0);
	}
	element_clear(e0);
    } else {
	mpz_init(p->tateexp);
	mpz_sub_ui(p->tateexp, p->Fqk->order, 1);
	mpz_divexact(p->tateexp, p->tateexp, pairing->r);
    }

    p->mapbase = Fq_to_Fqd_to_Fqk;

    cc_init_map_curve(p->Etwist, p->Eq, p->Fqd, element_field_to_poly);
    twist_curve(p->Etwist);
    element_init(p->nqrinv, p->Fqd);
    element_invert(p->nqrinv, field_get_nqr(p->Fqd));
    element_init(p->nqrinv2, p->Fqd);
    //TODO: element_square
    element_mul(p->nqrinv2, p->nqrinv, p->nqrinv);

    pairing->G1 = malloc(sizeof(field_t));
    pairing->G2 = malloc(sizeof(field_t));

    field_init_curve_group(pairing->G1, p->Eq, param->h);
    mpz_init(z);
    mpz_set_si(z, 1);
    field_init_curve_group(pairing->G2, p->Etwist, z);
    mpz_clear(z);
    p->k = param->k;
    pairing->GT = p->Fqk;
    //pairing->phi = trace;

    element_clear(a);
    element_clear(b);
}

void pairing_init_cc_param(pairing_t pairing, cc_param_t param)
{
    if (0) {
	pairing_init_cc_param_odd_k(pairing, param);
    } else {
	if (param->k % 2) pairing_init_cc_param_odd_k(pairing, param);
	else pairing_init_cc_param_even_k(pairing, param);
    }
}

static void phi_identity(element_ptr out, element_ptr in, pairing_ptr pairing)
{
    element_set(out, in);
}

static void solinas_pairing(element_ptr out, element_ptr in1, element_ptr in2,
	pairing_t pairing)
//in1, in2 are from E(F_q), out from F_q^2
{
    solinas_pairing_data_ptr p = pairing->data;
    point_t V, V1;
    element_t f, f0, f1;
    element_t a, b, c;
    element_t e0;
    int i, n;
    element_ptr Qx = ((point_ptr) in2->data)->x;
    element_ptr Qy = ((point_ptr) in2->data)->y;

    void do_tangent(void) {
	//a = -slope_tangent(V.x, V.y);
	//b = 1;
	//c = -(V.y + aV.x);
	//but we multiply by -2*V.y to avoid division so:
	//a = -(Vx (Vx + Vx + Vx) + cc->a)
	//b = 2 * Vy
	//c = -(2 Vy^2 + a Vx);
	element_ptr Vx = V->x;
	element_ptr Vy = V->y;
	element_add(a, Vx, Vx);
	element_add(a, a, Vx);
	element_mul(a, a, Vx);
	element_set1(b);
	element_add(a, a, b);
	element_neg(a, a);

	element_add(b, Vy, Vy);

	element_mul(e0, b, Vy);
	element_mul(c, a, Vx);
	element_add(c, c, e0);
	element_neg(c, c);

	//we'll map Q via (x,y) --> (-x, iy)
	//hence a Qx + c = -a Qx + c is real while
	//(b Qy) = b Qy i is purely imaginary.
	element_mul(a, a, Qx);
	element_sub(fi_re(f0), c, a);
	element_mul(fi_im(f0), b, Qy);
	element_mul(f, f, f0);
    }

    void do_line(point_ptr A, point_ptr B) {
	//a = -(B.y - A.y) / (B.x - A.x);
	//b = 1;
	//c = -(A.y + a * A.x);
	//but we'll multiply by B.x - A.x to avoid division, so
	//a = -(By - Ay)
	//b = Bx - Ax
	//c = -(Ay b + a Ax);
	element_sub(a, A->y, B->y);
	element_sub(b, B->x, A->x);
	element_mul(e0, a, A->x);
	element_mul(c, b, A->y);
	element_add(c, c, e0);
	element_neg(c, c);

	//we'll map Q via (x,y) --> (-x, iy)
	//hence a Qx + c = -a Qx + c is real while
	//(b Qy) = b Qy i is purely imaginary.
	element_mul(a, a, Qx);
	element_sub(fi_re(f0), c, a);
	element_mul(fi_im(f0), b, Qy);
	element_mul(f, f, f0);
    }

   if (0) {
	//check against standard Miller's alogrithm
	fq_data_ptr fdp;
	element_t x, y;
	mpz_t exp;

	element_init(x, p->Fq2);
	element_init(y, p->Fq2);
	fdp = x->data;
	element_neg(fdp->x, Qx);
	element_set0(fdp->y);
	fdp = y->data;
	element_set(fdp->y, Qy);
	element_set0(fdp->x);

	cc_miller(out, pairing->r, in1->data, x, y, element_field_to_quadratic);

	mpz_init(exp);
	mpz_sub_ui(exp, p->Fq2->order, 1);
	mpz_divexact(exp, exp, pairing->r);
	element_pow(out, out, exp);
	mpz_clear(exp);
	printf("check: ");
	element_out_str(stdout, 0, out);
	printf("\n");

	element_clear(x);
	element_clear(y);
    }

    point_init(V, p->Eq);
    point_init(V1, p->Eq);
    point_set(V, in1->data);
    element_init(f, p->Fq2);
    element_init(f0, p->Fq2);
    element_init(f1, p->Fq2);
    element_set1(f);
    element_init(a, p->Fq);
    element_init(b, p->Fq);
    element_init(c, p->Fq);
    element_init(e0, p->Fq);
    n = p->exp1;
    for (i=0; i<n; i++) {
	//f = f^2 g_V,V(Q)
	//where g_V,V = tangent at V
	//TODO: implement element_square?
	element_mul(f, f, f);
	do_tangent();
	point_double(V, V);
    }
    if (p->sign1 < 0) {
	point_neg(V1, V);
	element_invert(f1, f);
    } else {
	point_set(V1, V);
	element_set(f1, f);
    }
    n = p->exp2;
    for (; i<n; i++) {
	element_mul(f, f, f);
	do_tangent();
	point_double(V, V);
    }

    element_mul(f, f, f1);
    do_line(V, V1);

    //Tate exponentiation
    //simpler but slower:
    //element_pow(out, f, p->tateexp);
    //use this trick instead:
    element_invert(f0, f);
    element_neg(fi_im(f), fi_im(f));
    element_mul(f, f, f0);
    element_pow(out, f, p->h);

    element_clear(f);
    element_clear(f0);
    element_clear(f1);
    point_clear(V);
    point_clear(V1);
    element_clear(a);
    element_clear(b);
    element_clear(c);
    element_clear(e0);
}

void pairing_init_solinas_param(pairing_t pairing, solinas_param_t param)
{
    element_t a, b;
    solinas_pairing_data_ptr p;

    p =	pairing->data = malloc(sizeof(solinas_pairing_data_t));
    p->exp2 = param->exp2;
    p->exp1 = param->exp1;
    p->sign1 = param->sign1;
    mpz_init(pairing->r);
    mpz_set(pairing->r, param->r);
    field_init_fp(pairing->Zr, pairing->r);
    pairing->map = solinas_pairing;
    //pairing->is_almost_coddh = cc_is_almost_coddh;

    field_init_fp(p->Fq, param->q);
    element_init(a, p->Fq);
    element_init(b, p->Fq);
    element_set1(a);
    element_set0(b);
    curve_init_cc_ab(p->Eq, a, b);
    element_clear(a);
    element_clear(b);

    field_init_fi(p->Fq2, p->Fq);

    mpz_init(p->h);
    mpz_set(p->h, param->h);

    pairing->G1 = malloc(sizeof(field_t));
    field_init_curve_group(pairing->G1, p->Eq, param->h);
    pairing->G2 = pairing->G1;
    pairing->phi = phi_identity;
    pairing->GT = p->Fq2;
}
