#include "poly.h"
#include "fieldquadratic.h"
#include "hilbert.h"
#include "c_param.h"
#include "param.h"

struct mnt_pairing_data_s {
    field_t Fq, Fqx, Fqk;
    curve_t Eq, Eqk;
    mpz_t tateexp;
    int k;
    fieldmap mapbase;
    element_ptr *xpowq;
};
typedef struct mnt_pairing_data_s mnt_pairing_data_t[1];
typedef struct mnt_pairing_data_s *mnt_pairing_data_ptr;

struct even_mnt_pairing_data_s {
    field_t Fq, Fqx, Fqd, Fqk;
    curve_t Eq, Etwist;
    element_t nqrinv, nqrinv2;
    mpz_t tateexp;
    int k;
    fieldmap mapbase;
    element_ptr *xpowq;
};
typedef struct even_mnt_pairing_data_s even_mnt_pairing_data_t[1];
typedef struct even_mnt_pairing_data_s *even_mnt_pairing_data_ptr;

void c_param_init(c_param_ptr param)
{
    mpz_init(param->q);
    mpz_init(param->n);
    mpz_init(param->h);
    mpz_init(param->r);
    mpz_init(param->a);
    mpz_init(param->b);
    mpz_init(param->nk);
    mpz_init(param->hk);
    param->k = 0;
    param->coeff = NULL;
    mpz_init(param->nqr);
}

void c_param_clear(c_param_ptr param)
{
    int d = param->k / 2;
    int i;
    mpz_clear(param->q);
    mpz_clear(param->n);
    mpz_clear(param->h);
    mpz_clear(param->r);
    mpz_clear(param->a);
    mpz_clear(param->b);
    mpz_clear(param->nk);
    mpz_clear(param->hk);
    mpz_clear(param->nqr);
    for (i=0; i<d; i++) {
	mpz_clear(param->coeff[i]);
    }
    free(param->coeff);
}

void c_param_out_str(FILE *stream, c_param_ptr p)
{
    int d = p->k / 2;
    int i;
    char s[80];
    param_out_type(stream, "c");
    param_out_mpz(stream, "q", p->q);
    param_out_mpz(stream, "n", p->n);
    param_out_mpz(stream, "h", p->h);
    param_out_mpz(stream, "r", p->r);
    param_out_mpz(stream, "a", p->a);
    param_out_mpz(stream, "b", p->b);
    param_out_int(stream, "k", p->k);
    param_out_mpz(stream, "nk", p->nk);
    param_out_mpz(stream, "hk", p->hk);
    for (i=0; i<d; i++) {
	sprintf(s, "coeff%d", i);
	param_out_mpz(stream, s, p->coeff[i]);
    }
    param_out_mpz(stream, "nqr", p->nqr);
}

void c_param_inp_str(c_param_ptr p, FILE *stream)
{
    symtab_t tab;
    char s[80];
    int i, d;

    symtab_init(tab);
    param_read(tab, stream);

    lookup_mpz(p->q, tab, "q");
    lookup_mpz(p->n, tab, "n");
    lookup_mpz(p->h, tab, "h");
    lookup_mpz(p->r, tab, "r");
    lookup_mpz(p->a, tab, "a");
    lookup_mpz(p->b, tab, "b");
    p->k = lookup_int(tab, "k");
    lookup_mpz(p->nk, tab, "nk");
    lookup_mpz(p->hk, tab, "hk");
    lookup_mpz(p->nqr, tab, "nqr");

    d = p->k / 2;
    p->coeff = realloc(p->coeff, sizeof(mpz_t) * d);
    for (i=0; i<d; i++) {
	sprintf(s, "coeff%d", i);
	mpz_init(p->coeff[i]);
	lookup_mpz(p->coeff[i], tab, s);
    }

    param_clear_tab(tab);
    symtab_clear(tab);
}
//
//assumes P is in the base field, Q in some field extension
static void cc_miller(element_t res, mpz_t q, point_t P,
	element_ptr Qx, element_ptr Qy, fieldmap mapbase)
{
    //collate divisions
    int m;
    element_t v, vd;
    point_t Z;
    element_t a, b, c;
    common_curve_ptr cc = P->curve->data;
    element_t t0;
    element_t e0, e1;

    void do_vertical(element_t e)
    {
	if (point_is_inf(Z)) {
	    return;
	}
	mapbase(e0, Z->x);
	element_sub(e0, Qx, e0);
	element_mul(e, e, e0);
    }

    void do_tangent(element_t e)
    {
	//a = -slope_tangent(Z.x, Z.y);
	//b = 1;
	//c = -(Z.y + a * Z.x);
	//but we multiply by 2*Z.y to avoid division

	//a = -Zx * (3 Zx + twicea_2) - a_4;
	//Common curves: a2 = 0 (and cc->a is a_4), so
	//a = -(3 Zx^2 + cc->a)
	//b = 2 * Zy
	//c = -(2 Zy^2 + a Zx);
	element_ptr Zx = Z->x;
	element_ptr Zy = Z->y;

	if (point_is_inf(Z)) {
	    return;
	}

	if (element_is0(Zy)) {
	    //order 2
	    do_vertical(e);
	    return;
	}

	element_square(a, Zx);
	element_mul_si(a, a, 3);
	element_add(a, a, cc->a);
	element_neg(a, a);

	element_add(b, Zy, Zy);

	element_mul(t0, b, Zy);
	element_mul(c, a, Zx);
	element_add(c, c, t0);
	element_neg(c, c);

	//TODO: use poly_mul_constant?
	mapbase(e0, a);
	element_mul(e0, e0, Qx);
	mapbase(e1, b);
	element_mul(e1, e1, Qy);
	element_add(e0, e0, e1);
	mapbase(e1, c);
	element_add(e0, e0, e1);
	element_mul(e, e, e0);
    }

    void do_line(element_ptr e)
    {
	//a = -(B.y - A.y) / (B.x - A.x);
	//b = 1;
	//c = -(A.y + a * A.x);
	//but we'll multiply by B.x - A.x to avoid division

	element_ptr Ax = Z->x;
	element_ptr Ay = Z->y;
	element_ptr Bx = P->x;
	element_ptr By = P->y;

	//we assume B is never O
	if (point_is_inf(Z)) {
	    do_vertical(e);
	    return;
	}

	if (!element_cmp(Ax, Bx)) {
	    if (!element_cmp(Ay, By)) {
		do_tangent(e);
		return;
	    }
	    do_vertical(e);
	    return;
	}

	element_sub(b, Bx, Ax);
	element_sub(a, Ay, By);
	element_mul(t0, b, Ay);
	element_mul(c, a, Ax);
	element_add(c, c, t0);
	element_neg(c, c);

	mapbase(e0, a);
	element_mul(e0, e0, Qx);
	mapbase(e1, b);
	element_mul(e1, e1, Qy);
	element_add(e0, e0, e1);
	mapbase(e1, c);
	element_add(e0, e0, e1);
	element_mul(e, e, e0);
    }

    element_init(a, P->curve->field);
    element_init(b, P->curve->field);
    element_init(c, P->curve->field);
    element_init(t0, P->curve->field);
    element_init(e0, res->field);
    element_init(e1, res->field);

    element_init(v, res->field);
    element_init(vd, res->field);
    point_init(Z, P->curve);

    point_set(Z, P);

    element_set1(v);
    element_set1(vd);
    m = mpz_sizeinbase(q, 2) - 2;

    while(m >= 0) {
	element_square(v, v);
	element_square(vd, vd);
	do_tangent(v);
	point_double(Z, Z);
	do_vertical(vd);
	if (mpz_tstbit(q, m)) {
	    do_line(v);
	    point_add(Z, Z, P);
	    do_vertical(vd);
	}
	m--;
    }

    element_invert(vd, vd);
    element_mul(res, v, vd);

    element_clear(v);
    element_clear(vd);
    point_clear(Z);
    element_clear(a);
    element_clear(b);
    element_clear(c);
    element_clear(t0);
    element_clear(e0);
    element_clear(e1);
}

static void point_proj_double(point_t R, element_t Rz, point_t P, element_t z)
{
    common_curve_ptr cc = P->curve->data;
    element_t e0, e1, e2, e3;
    element_init(e0, z->field);
    element_init(e1, z->field);
    element_init(e2, z->field);
    element_init(e3, z->field);
    element_ptr x = P->x;
    element_ptr y = P->y;
    //e0 = 3x^2 + (cc->a) z^4
    element_square(e0, x);
    element_mul_si(e0, e0, 3);
    element_square(e1, z);
    element_square(e1, e1);
    element_mul(e1, e1, cc->a);
    element_add(e0, e0, e1);

    //z_out = 2 y z
    element_mul(Rz, y, z);
    element_mul_si(Rz, Rz, 2);

    //e1 = 4 x y^2
    element_square(e2, y);
    element_mul(e1, x, e2);
    element_mul_si(e1, e1, 4);

    //x_out = e0^2 - 2 e1
    element_mul_si(e3, e1, 2);
    element_square(R->x, e0);
    element_sub(R->x, R->x, e3);

    //e2 = 8y^4
    element_square(e2, e2);
    element_mul_si(e2, e2, 8);

    //y_out = e0(e1 - x_out) - e2
    element_sub(e1, e1, R->x);
    element_mul(e0, e0, e1);
    element_sub(R->y, e0, e2);

    R->inf_flag = 0;

    element_clear(e0);
    element_clear(e1);
    element_clear(e2);
    element_clear(e3);
}

//convert P from weighted projective (Jacobian) to affine
//i.e. (X, Y, Z) --> (X/Z^2, Y/Z^3)
//also sets z to 1
static void point_to_affine(point_ptr P, element_t z)
{
    element_t e0;

    element_init(e0, z->field);
    element_invert(z, z);
    element_square(e0, z);
    element_mul(P->x, P->x, e0);
    element_mul(e0, e0, z);
    element_mul(P->y, P->y, e0);
    element_set1(z);

    element_clear(e0);
}

static void cc_tatepower(element_ptr out, element_ptr in, pairing_t pairing)
{
    mnt_pairing_data_ptr p = pairing->data;
    element_pow(out, in, p->tateexp);
}

static void cc_pairing(element_ptr out, element_ptr in1, element_ptr in2,
	pairing_t pairing)
{
    mnt_pairing_data_ptr p = pairing->data;
    point_ptr Q = in2->data;
    cc_miller(out, pairing->r, in1->data, Q->x, Q->y, p->mapbase);
    cc_tatepower(out, out, pairing);
}

static int cc_is_almost_coddh_odd_k(element_ptr a, element_ptr b,
	element_ptr c, element_ptr d,
	pairing_t pairing)
{
    int res = 0;
    element_t t0, t1, t2;

    element_init(t0, pairing->GT);
    element_init(t1, pairing->GT);
    element_init(t2, pairing->GT);
    mnt_pairing_data_ptr p = pairing->data;
    point_ptr Pc = c->data;
    point_ptr Pd = d->data;
    cc_miller(t0, pairing->r, a->data, Pd->x, Pd->y, p->mapbase);
    cc_miller(t1, pairing->r, b->data, Pc->x, Pc->y, p->mapbase);
    cc_tatepower(t1, t1, pairing);
    cc_tatepower(t0, t0, pairing);
    element_mul(t2, t0, t1);
    if (element_is1(t2)) {
	//g, g^x, h, h^-x case
	res = 1;
    } else {
	element_invert(t1, t1);
	element_mul(t2, t0, t1);
	if (element_is1(t2)) {
	    //g, g^x, h, h^x case
	    res = 1;
	}
    }
    element_clear(t0);
    element_clear(t1);
    element_clear(t2);
    return res;
}

static void trace(element_ptr out, element_ptr in, pairing_ptr pairing)
{
    int i;
    point_ptr p = in->data;
    point_ptr r = out->data;
    point_t q;
    mnt_pairing_data_ptr mpdp = pairing->data;

    point_init(q, p->curve);

    point_set(q, p);
    point_set(r, p);

    for (i=1; i<mpdp->k; i++) {
	cc_frobenius(q, q, mpdp->Fq->order);
	point_add(r, r, q);
    }
    point_clear(q);
}

static void pairing_init_c_param_odd_k(pairing_t pairing, c_param_t param)
{
    mnt_pairing_data_ptr p;
    element_t a, b;
    element_t irred;
    mpz_t z;

    mpz_init(pairing->r);
    mpz_set(pairing->r, param->r);
    field_init_fp(pairing->Zr, pairing->r);
    pairing->map = cc_pairing;
    pairing->is_almost_coddh = cc_is_almost_coddh_odd_k;

    p =	pairing->data = malloc(sizeof(mnt_pairing_data_t));
    field_init_fp(p->Fq, param->q);
    element_init(a, p->Fq);
    element_init(b, p->Fq);
    element_set_mpz(a, param->a);
    element_set_mpz(b, param->b);
    curve_init_cc_ab(p->Eq, a, b);

    field_init_poly(p->Fqx, p->Fq);
    element_init(irred, p->Fqx);
    do {
	poly_random_monic(irred, param->k);
    } while (!poly_is_irred(irred));
    field_init_polymod(p->Fqk, irred);
    element_clear(irred);

    mpz_init(p->tateexp);
    mpz_sub_ui(p->tateexp, p->Fqk->order, 1);
    mpz_divexact(p->tateexp, p->tateexp, pairing->r);

    p->mapbase = ((polymod_field_data_ptr) p->Fqk->data)->mapbase;

    cc_init_map_curve(p->Eqk, p->Eq, p->Fqk, p->mapbase);

    pairing->G1 = malloc(sizeof(field_t));
    pairing->G2 = malloc(sizeof(field_t));

    field_init_curve_group(pairing->G1, p->Eq, param->h);
    mpz_init(z);
    mpz_set_si(z, 1);
    field_init_curve_group(pairing->G2, p->Eqk, z);
    mpz_clear(z);
    p->k = param->k;
    pairing->GT = p->Fqk;
    pairing->phi = trace;

    element_clear(a);
    element_clear(b);
}

static void cc_miller_no_denom(element_t res, mpz_t q, point_t P,
	element_ptr Qx, element_ptr Qy, fieldmap mapbase)
{
    //collate divisions
    int m;
    element_t v;
    point_t Z;
    element_t a, b, c;
    common_curve_ptr cc = P->curve->data;
    element_t t0;
    element_t e0, e1;
    element_t z;

    void do_vertical(void)
    {
	if (point_is_inf(Z)) {
	    return;
	}
	mapbase(e0, Z->x);
	element_sub(e0, Qx, e0);
	element_mul(v, v, e0);
    }

    void do_tangent(void)
    {
	//a = -slope_tangent(Z.x, Z.y);
	//b = 1;
	//c = -(Z.y + a * Z.x);
	//but we multiply by 2*Z.y to avoid division

	//a = -Zx * (Zx + Zx + Zx + twicea_2) - a_4;
	//Common curves: a2 = 0 (and cc->a is a_4), so
	//a = -(3 Zx^2 + cc->a)
	//b = 2 * Zy
	//c = -(2 Zy^2 + a Zx);
	element_ptr Zx = Z->x;
	element_ptr Zy = Z->y;

	if (point_is_inf(Z)) {
	    return;
	}

	if (element_is0(Zy)) {
	    //order 2
	    do_vertical();
	    return;
	}

	element_square(a, Zx);
	element_mul_si(a, a, 3);
	element_add(a, a, cc->a);
	element_neg(a, a);

	element_add(b, Zy, Zy);

	element_mul(t0, b, Zy);
	element_mul(c, a, Zx);
	element_add(c, c, t0);
	element_neg(c, c);

	//TODO: use poly_mul_constant?
	mapbase(e0, a);
	element_mul(e0, e0, Qx);
	mapbase(e1, b);
	element_mul(e1, e1, Qy);
	element_add(e0, e0, e1);
	mapbase(e1, c);
	element_add(e0, e0, e1);
	element_mul(v, v, e0);
    }

    void do_line(void)
    {
	//a = -(B.y - A.y) / (B.x - A.x);
	//b = 1;
	//c = -(A.y + a * A.x);
	//but we'll multiply by B.x - A.x to avoid division

	element_ptr Ax = Z->x;
	element_ptr Ay = Z->y;
	element_ptr Bx = P->x;
	element_ptr By = P->y;

	//we assume B is never O
	if (point_is_inf(Z)) {
	    do_vertical();
	    return;
	}

	if (!element_cmp(Ax, Bx)) {
	    if (!element_cmp(Ay, By)) {
		do_tangent();
		return;
	    }
	    do_vertical();
	    return;
	}

	element_sub(b, Bx, Ax);
	element_sub(a, Ay, By);
	element_mul(t0, b, Ay);
	element_mul(c, a, Ax);
	element_add(c, c, t0);
	element_neg(c, c);

	mapbase(e0, a);
	element_mul(e0, e0, Qx);
	mapbase(e1, b);
	element_mul(e1, e1, Qy);
	element_add(e0, e0, e1);
	mapbase(e1, c);
	element_add(e0, e0, e1);
	element_mul(v, v, e0);
    }

    element_init(a, P->curve->field);
    element_init(b, P->curve->field);
    element_init(c, P->curve->field);
    element_init(t0, P->curve->field);
    element_init(e0, res->field);
    element_init(e1, res->field);
    element_init(z, a->field);
    element_set1(z);

    element_init(v, res->field);
    point_init(Z, P->curve);

    point_set(Z, P);

    element_set1(v);
    m = mpz_sizeinbase(q, 2) - 2;

    while(m >= 0) {
	element_square(v, v);
	do_tangent();
	point_double(Z, Z);
	if (mpz_tstbit(q, m)) {
	    do_line();
	    point_add(Z, Z, P);
	}
	m--;
    }

    element_set(res, v);

    element_clear(v);
    point_clear(Z);
    element_clear(a);
    element_clear(b);
    element_clear(c);
    element_clear(t0);
    element_clear(e0);
    element_clear(e1);
    element_clear(z);
}

static void cc_tatepower_even_k(element_ptr out, element_ptr in, pairing_t pairing)
{
    even_mnt_pairing_data_ptr p = pairing->data;
    if (p->k == 6) {
	element_t e0, e1, e2, e3;
	element_init(e0, p->Fqk);
	element_init(e1, p->Fqd);
	element_init(e2, p->Fqd);
	element_init(e3, p->Fqk);
	element_ptr e0re = fi_re(e0);
	element_ptr e0im = fi_im(e0);
	element_t *inre = fi_re(in)->data;
	element_t *inim = fi_im(in)->data;
	void subexpr(element_ptr e) {
	    element_set0(e0);
	    element_set(((element_t *) e0re->data)[0], inre[0]);
	    element_set(((element_t *) e0im->data)[0], inim[0]);
	    polymod_const_mul(e2, inre[1], e);
	    element_add(e0re, e0re, e2);
	    polymod_const_mul(e2, inim[1], e);
	    element_add(e0im, e0im, e2);
	    element_square(e1, e);
	    polymod_const_mul(e2, inre[2], e1);
	    element_add(e0re, e0re, e2);
	    polymod_const_mul(e2, inim[2], e1);
	    element_add(e0im, e0im, e2);
	}
	subexpr(p->xpowq[4]);
	element_set(e3, e0);
	subexpr(p->xpowq[3]);
	element_neg(e0im, e0im);
	element_mul(e3, e3, e0);
	subexpr(p->xpowq[1]);
	element_neg(e0im, e0im);
	element_mul(e0, e0, in);
	element_invert(e0, e0);
	element_mul(out, e3, e0);
	element_pow(out, out, p->tateexp);
	element_clear(e0);
	element_clear(e1);
	element_clear(e2);
	element_clear(e3);
    } else {
	element_pow(out, in, p->tateexp);
    }
}

static void cc_pairing_even_k(element_ptr out, element_ptr in1, element_ptr in2,
	pairing_t pairing)
{
    point_ptr Qbase = in2->data;
    element_t x, y;
    even_mnt_pairing_data_ptr p = pairing->data;

    element_init(x, out->field);
    element_init(y, out->field);
    //map from twist: (x, y) --> (v^-1 x, v^-(3/2) y)
    //where v is the quadratic nonresidue used to construct the twist
    element_mul(fi_re(x), Qbase->x, p->nqrinv);
    //v^-3/2 = v^-2 * v^1/2
    element_mul(fi_im(y), Qbase->y, p->nqrinv2);
    cc_miller_no_denom(out, pairing->r, in1->data, x, y, p->mapbase);
    cc_tatepower_even_k(out, out, pairing);
    element_clear(x);
    element_clear(y);
}

static int cc_is_almost_coddh_even_k(element_ptr a, element_ptr b,
	element_ptr c, element_ptr d,
	pairing_t pairing)
{
    int res = 0;
    element_t t0, t1, t2;
    element_t cx, cy;
    element_t dx, dy;
    even_mnt_pairing_data_ptr p = pairing->data;

    element_init(cx, p->Fqk);
    element_init(cy, p->Fqk);
    element_init(dx, p->Fqk);
    element_init(dy, p->Fqk);

    element_init(t0, pairing->GT);
    element_init(t1, pairing->GT);
    element_init(t2, pairing->GT);
    point_ptr Pc = c->data;
    point_ptr Pd = d->data;
    //map from twist: (x, y) --> (v^-1 x, v^-(3/2) y)
    //where v is the quadratic nonresidue used to construct the twist
    element_mul(fi_re(cx), Pc->x, p->nqrinv);
    element_mul(fi_re(dx), Pd->x, p->nqrinv);
    //v^-3/2 = v^-2 * v^1/2
    element_mul(fi_im(cy), Pc->y, p->nqrinv2);
    element_mul(fi_im(dy), Pd->y, p->nqrinv2);

    cc_miller_no_denom(t0, pairing->r, a->data, dx, dy, p->mapbase);
    cc_miller_no_denom(t1, pairing->r, b->data, cx, cy, p->mapbase);
    cc_tatepower_even_k(t0, t0, pairing);
    cc_tatepower_even_k(t1, t1, pairing);
    element_mul(t2, t0, t1);
    if (element_is1(t2)) {
	//g, g^x, h, h^-x case
	res = 1;
    } else {
	element_invert(t1, t1);
	element_mul(t2, t0, t1);
	if (element_is1(t2)) {
	    //g, g^x, h, h^x case
	    res = 1;
	}
    }
    element_clear(t0);
    element_clear(t1);
    element_clear(t2);
    return res;
}

static void Fq_to_Fqd_to_Fqk(element_t out, element_t in)
{
    element_field_to_polymod(fi_re(out), in);
    element_set0(fi_im(out));
}

static void pairing_init_c_param_even_k(pairing_t pairing, c_param_t param)
{
    even_mnt_pairing_data_ptr p;
    element_t a, b;
    element_t irred;
    mpz_t one;
    int d = param->k / 2;
    int i;

    mpz_init(pairing->r);
    mpz_set(pairing->r, param->r);
    field_init_fp(pairing->Zr, pairing->r);
    pairing->map = cc_pairing_even_k;
    pairing->is_almost_coddh = cc_is_almost_coddh_even_k;

    p =	pairing->data = malloc(sizeof(even_mnt_pairing_data_t));
    field_init_fp(p->Fq, param->q);
    element_init(a, p->Fq);
    element_init(b, p->Fq);
    element_set_mpz(a, param->a);
    element_set_mpz(b, param->b);
    curve_init_cc_ab(p->Eq, a, b);

    field_init_poly(p->Fqx, p->Fq);
    element_init(irred, p->Fqx);
    poly_alloc(irred, d + 1);
    for (i=0; i<d; i++) {
	element_set_mpz(poly_coeff(irred, i), param->coeff[i]);
    }
    element_set1(poly_coeff(irred, i));

    field_init_polymod(p->Fqd, irred);
    element_clear(irred);

    p->Fqd->nqr = malloc(sizeof(element_t));
    element_init(p->Fqd->nqr, p->Fqd);
    element_set_mpz(((element_t *) p->Fqd->nqr->data)[0], param->nqr);

    field_init_quadratic(p->Fqk, p->Fqd);

    if (param->k == 6) {
	element_t e0;
	mpz_ptr q = param->q;
	mpz_ptr z = p->tateexp;
	mpz_init(z);
	mpz_mul(z, q, q);
	mpz_sub(z, z, q);
	mpz_add_ui(z, z, 1);
	mpz_divexact(z, z, pairing->r);

	p->xpowq = malloc(sizeof(element_ptr) * 5);
	element_init(e0, p->Fqd);
	element_set1(((element_t *) e0->data)[1]);
	for (i=1; i<=4; i++) {
	    element_ptr e = p->xpowq[i] = malloc(sizeof(element_t));
	    element_init(e, p->Fqd);
	    element_pow(e0, e0, q);
	    element_set(e, e0);
	}
	element_clear(e0);
    } else {
	mpz_init(p->tateexp);
	mpz_sub_ui(p->tateexp, p->Fqk->order, 1);
	mpz_divexact(p->tateexp, p->tateexp, pairing->r);
    }

    p->mapbase = Fq_to_Fqd_to_Fqk;

    cc_init_map_curve(p->Etwist, p->Eq, p->Fqd, element_field_to_polymod);
    twist_curve(p->Etwist);
    element_init(p->nqrinv, p->Fqd);
    element_invert(p->nqrinv, field_get_nqr(p->Fqd));
    element_init(p->nqrinv2, p->Fqd);
    element_square(p->nqrinv2, p->nqrinv);

    pairing->G1 = malloc(sizeof(field_t));
    pairing->G2 = malloc(sizeof(field_t));

    field_init_curve_group(pairing->G1, p->Eq, param->h);
    mpz_init(one);
    mpz_set_si(one, 1);
    field_init_curve_group(pairing->G2, p->Etwist, one);
    mpz_clear(one);
    p->k = param->k;
    pairing->GT = p->Fqk;
    //pairing->phi = trace;

    element_clear(a);
    element_clear(b);
}

void pairing_init_c_param(pairing_t pairing, c_param_t param)
{
    if (0) {
	pairing_init_c_param_odd_k(pairing, param);
    } else {
	if (param->k % 2) pairing_init_c_param_odd_k(pairing, param);
	else pairing_init_c_param_even_k(pairing, param);
    }
}

static void compute_cm_curve(c_param_ptr param, cm_info_ptr cm)
    //computes a curve and sets fp to the field it is defined over
    //using the complex multiplication method, where cm holds
    //the appropriate information (e.g. discriminant, field order)
{
    darray_t coefflist;
    element_t hp, root;
    field_t fp, fpx;
    int i, n;
    curve_t cc;

    field_init_fp(fp, cm->q);
    field_init_poly(fpx, fp);
    element_init(hp, fpx);

    darray_init(coefflist);

    hilbert_poly(coefflist, cm->D);

    if (0) {
	n = coefflist->count;
	printf("x^%d", n - 1);
	printf("\n");
	for (i=n-2; i>=0; i--) {
	    mpz_out_str(stdout, 0, coefflist->item[i]);
	    if (i) {
		if (i == 1) {
		    printf("x");
		} else {
		    printf("x^%d", i);
		}
	    }
	    printf("\n");
	}
    }

    n = coefflist->count;
    poly_alloc(hp, n);
    for (i=0; i<n; i++) {
	element_set_mpz(poly_coeff(hp, i), coefflist->item[i]);
    }
    //TODO: memory leak: clear elements of coefflist
    darray_clear(coefflist);
    //TODO: remove x = 0, 1728 roots
    //TODO: what if there's no roots?
    //printf("hp ");
    //element_out_str(stdout, 0, hp);
    //printf("\n");
    element_init(root, fp);
    findroot(root, hp);
    //printf("root = ");
    //element_out_str(stdout, 0, root);
    //printf("\n");
    element_clear(hp);
    field_clear(fpx);

    //the root is the j-invariant of our desired curve
    curve_init_cc_j(cc, root);
    element_clear(root);

    //we may need to twist it however
    {
	point_t P;

	//pick a random point P and see if it has the right order
	point_init(P, cc);
	point_random(P);
	point_mul(P, cm->n, P);
	//printf("P = ");
	//point_out_str(stdout, 0, P);
	//printf("\n");
	//if not, we twist the curve
	if (!point_is_inf(P)) {
	    twist_curve(cc);
	}
	point_clear(P);
    }

    mpz_set(param->q, cm->q);
    mpz_set(param->n, cm->n);
    mpz_set(param->h, cm->h);
    mpz_set(param->r, cm->r);
    mpz_set(param->a, ((common_curve_ptr) cc->data)->a->data);
    mpz_set(param->b, ((common_curve_ptr) cc->data)->b->data);
    param->k = cm->k;
    {
	mpz_t z;
	mpz_init(z);
	//compute order of curve in F_q^k
	//n = q - t + 1 hence t = q - n + 1
	mpz_sub(z, param->q, param->n);
	mpz_add_ui(z, z, 1);
	compute_trace_n(z, param->q, z, param->k);
	mpz_pow_ui(param->nk, param->q, param->k);
	mpz_sub_ui(z, z, 1);
	mpz_sub(param->nk, param->nk, z);
	mpz_mul(z, param->r, param->r);
	mpz_divexact(param->hk, param->nk, z);
	mpz_clear(z);
    }
    curve_clear(cc);
    field_clear(fp);
}

void c_param_from_cm(c_param_t param, cm_info_ptr cm)
{
    //TODO: handle odd case as well?
    field_t Fq, Fqx, Fqd;
    element_t irred, nqr;
    int d = cm->k / 2;
    int i;

    compute_cm_curve(param, cm);

    field_init_fp(Fq, param->q);
    field_init_poly(Fqx, Fq);
    element_init(irred, Fqx);
    do {
	poly_random_monic(irred, d);
    } while (!poly_is_irred(irred));
    field_init_polymod(Fqd, irred);

    //find a quadratic nonresidue of Fqd lying in Fq
    //TODO: we're assuming there exists an element in Fq
    //that is not a square in Fqd
    element_init(nqr, Fqd);
    do {
	element_random(((element_t *) nqr->data)[0]);
    } while (element_is_sqr(nqr));

    param->coeff = realloc(param->coeff, sizeof(mpz_t) * d);

    for (i=0; i<d; i++) {
	mpz_init(param->coeff[i]);
	mpz_set(param->coeff[i], poly_coeff(irred, i)->data);
    }
    mpz_set(param->nqr, ((element_t *) nqr->data)[0]->data);

    element_clear(nqr);
    element_clear(irred);

    field_clear(Fqx);
    field_clear(Fqd);
    field_clear(Fq);
}
